unit DIBCanvas;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, Grafix,
  ColorPalette;

type

  TEdge = class( TObject )
  private
  protected
  public
    X: single;
    increment: single;
    YCutoff: integer;
  end;

  TDIBCanvas = class( TPersistent )
  private
    FWidth: integer;
    FHeight: integer;
    nDummy: integer;
    FOrient: TDIBOrientation;
    FBrush: byte;
    FPen: byte;
    FColorPalette: TColorPalette;
    FTrans: byte;
    FBits: pointer;
    ptCursor: TPoint;
    lstEdges: TList;
    procedure clearEdgeTable;
  protected
    function getClipRect: TRect;
    function getSize: integer;
    procedure setWidth( n: integer );
    function getPixels( X: integer; Y: integer ): byte;
    procedure setPixels( X: integer; Y: integer; Value: byte );
    function getPixelsClip( X: integer; Y: integer ): byte;
    procedure setPixelsClip( X: integer; Y: integer; Value: byte );
    function getAllocated: boolean;
  public
    constructor Create;
    destructor Destroy; override;
    procedure Allocate;
    procedure DeAllocate;
    procedure Circle( X, Y, Radius: integer );
    procedure CopyRect( Dest: TRect; Canvas: TDIBCanvas; Source: TRect; Transparent: boolean );
    procedure CopyRectQuad( Dest: TRect; Canvas: TDIBCanvas; Source: TRect; Transparent: boolean; QuadAngle: TQuadAngle );
    procedure Draw( X, Y: integer; DIBCanvas: TDIBCanvas );
    procedure FillCircle( X, Y, Radius: integer );
    procedure FillPolygon( X, Y: integer; poly: TPolygon );
    procedure FillRect( rect: TRect );
    procedure LineTo( X: integer; Y: integer );
    procedure MoveTo( X: integer; Y: integer );
    procedure Polygon( X, Y: integer; poly: TPolygon );
    procedure Rectangle( X1, Y1, X2, Y2: integer );
    property Allocated: boolean read getAllocated;
    property Bits: pointer read FBits write FBits;
    property ClipRect: TRect read getClipRect;
    property ColorPalette: TColorPalette read FColorPalette write FColorPalette;
    property Pixels[X: integer; Y: integer]: byte read getPixels write setPixels;
    property PixelsClipped[X: integer; Y: integer]: byte read getPixelsClip write setPixelsClip;
  published
    property BrushColorIndex: byte read FBrush write FBrush;
    property Height: integer read FHeight write FHeight;
    property Orientation: TDIBOrientation read FOrient write FOrient;
    property PenColorIndex: byte read FPen write FPen;
    property Size: integer read getSize write nDummy;
    property TransparentColorIndex: byte read FTrans write FTrans;
    property Width: integer read FWidth write setWidth;
  end;

implementation

constructor TDIBCanvas.Create;
begin
  lstEdges := TList.Create;
end;

destructor TDIBCanvas.Destroy;
begin
  DeAllocate;
  clearEdgeTable;
  lstEdges.Free;
  inherited Destroy;
end;

function TDIBCanvas.getClipRect: TRect;
begin
  Result := Rect( 0, 0, FWidth - 1, FHeight - 1 );
end;

function TDIBCanvas.getSize;
begin
  Result := FWidth * FHeight;
end;

function TDIBCanvas.getPixels( x, y: integer ): byte;
begin
  if FOrient = orTopDown then
    Result := PByteArray( FBits )^[y * FWidth + x]
  else
    Result := PByteArray( FBits )^[( FHeight - y - 1 ) * FWidth + x];
end;

procedure TDIBCanvas.setPixels( X: integer; Y: integer; Value: byte );
begin
  if FOrient = orTopDown then
    PByteArray( FBits )^[y * FWidth + x] := Value
  else
    PByteArray( FBits )^[( FHeight - y - 1 ) * FWidth + x] := Value;
end;

function TDIBCanvas.getPixelsClip( x, y: integer ): byte;
begin
  Result := 0;
  if PtInRect( ClipRect, Point( x, y ) ) then
    Result := Pixels[x, y];
end;

procedure TDIBCanvas.setPixelsClip( X: integer; Y: integer; Value: byte );
begin
  if PtInRect( ClipRect, Point( x, y ) ) then
    Pixels[x, y] := Value;
end;

procedure TDIBCanvas.setWidth( n: integer );
begin
  FWidth := (((n + 3) div 4) * 4);
end;

procedure TDIBCanvas.MoveTo( x, y: integer );
begin
  ptCursor := Point( x, y );
end;

procedure TDIBCanvas.LineTo( x, y: integer );
var
  x1_, y1_, x2_, y2_: integer;
begin
  x1_ := ptCursor.X;
  y1_ := ptCursor.Y;
  x2_ := x;
  y2_ := y;
  if ClipLine( x1_, y1_, x2_, y2_, ClipRect ) then
  begin
    if FOrient = orBottomUp then
    begin
      y1_ := FHeight - y1_ - 1;
      y2_ := FHeight - y2_ - 1;
    end;
    DrawLine( FBits, x1_, y1_, x2_, y2_, FWidth, FPen );
  end;
  ptCursor := Point( x, y );
end;

procedure TDIBCanvas.Rectangle( X1, Y1, X2, Y2: integer );
begin
  MoveTo( x1, y1 );
  LineTo( x2, y1 );
  LineTo( x2, y2 );
  LineTo( x1, y2 );
  LineTo( x1, y1 );
end;

procedure TDIBCanvas.FillRect( rect: TRect );
begin
  FillBlock( FBits, ( rect.Right - rect.Left + 1 ), ( rect.Bottom - rect.Top + 1 ),
    FWidth, FHeight, rect.Left, rect.Top, FBrush, FOrient );
end;

procedure TDIBCanvas.CopyRect( Dest: TRect; Canvas: TDIBCanvas; Source: TRect; Transparent: boolean );
begin
  if Transparent then
    CopyBlockTrans( Canvas.FBits, FBits, Canvas.Width, Canvas.Height,
      (Source.Right-Source.Left), (Source.Bottom-Source.Top), Source.Left, Source.Top,
      Width, Height, Dest.Left, Dest.Top, Canvas.TransparentColorIndex,
      Canvas.Orientation, Orientation )
  else
    CopyBlock( Canvas.FBits, FBits, Canvas.Width, Canvas.Height,
      (Source.Right-Source.Left), (Source.Bottom-Source.Top), Source.Left, Source.Top,
      Width, Height, Dest.Left, Dest.Top,
      Canvas.Orientation, Orientation );
end;

procedure TDIBCanvas.CopyRectQuad( Dest: TRect; Canvas: TDIBCanvas; Source: TRect; Transparent: boolean; QuadAngle: TQuadAngle );
begin
  CopyBlockTransQuad( Canvas.FBits, FBits, Canvas.Width, Canvas.Height,
      (Source.Right-Source.Left), (Source.Bottom-Source.Top), Source.Left, Source.Top,
      Width, Height, Dest.Left, Dest.Top, Canvas.TransparentColorIndex,
      Canvas.Orientation, Orientation, QuadAngle );
end;

procedure TDIBCanvas.Circle( X, Y, Radius: integer );
begin
  if Orientation = orBottomUp then
    Grafix.DrawCircle( FBits, X, Height - Y, Radius, Width, Height, FPen )
  else
    Grafix.DrawCircle( FBits, X, Y, Radius, Width, Height, FPen );
end;

procedure TDIBCanvas.FillCircle( X, Y, Radius: integer );
var
  x2, y2: integer;
  FillScan: boolean;
  State: integer;
begin
  Circle( X, Y, Radius );
  for y2 := 0 to Height - 1 do
  begin
    FillScan := false;
    State := 1;
    for x2 := 0 to Width - 1 do
    begin
      case State of
        1:
          if Pixels[x2,y2] = FPen then
            State := 2;
        2:
          if Pixels[x2,y2] <> FPen then
            State := 3;
        3:
          if Pixels[x2,y2] = FPen then
          begin
            FillScan := true;
            break;
          end;
      end;
    end;
    if FillScan then
    begin
      State := 1;
      for x2 := 0 to Width - 1 do
      begin
        case State of
          1:
            if Pixels[x2,y2] = FPen then
              State := 2;
          2:
            if Pixels[x2,y2] <> FPen then
            begin
              State := 3;
              Pixels[x2,y2] := FBrush;
            end;
          3:
            if Pixels[x2,y2] = FPen then
              break
            else
              Pixels[x2,y2] := FBrush;
        end;
      end;
    end;
  end;
end;

procedure TDIBCanvas.Allocate;
begin
  if FBits <> nil then
    DeAllocate;
  FBits := VirtualAlloc( nil, FWidth * FHeight, MEM_RESERVE or MEM_COMMIT, PAGE_READWRITE );
end;

procedure TDIBCanvas.DeAllocate;
begin
  if FBits <> nil then
    VirtualFree( FBits, FWidth * FHeight, MEM_RELEASE or MEM_DECOMMIT );
  FBits := nil;
end;

function TDIBCanvas.getAllocated: boolean;
begin
  Result := FBits <> nil;
end;

(*********************************************
Render a solid color, 2D filled polygon.
*********************************************)
procedure TDIBCanvas.fillPolygon( X, Y: integer; poly: TPolygon );
var
  vc, i, v: integer;
  miny, maxy: integer;
  y_: integer;
  edge, e1, e2: TEdge;
  xDiff: single;
  bSorted: boolean;
  rectCanvas: TRect;
  linex1, linex2, liney1, liney2: integer;
  pOut: Pbyte;
  function getAdjPoint( n: integer ): integer;
  begin
    if n = poly.NumVertex then
      Result := 0
    else if n < 0 then
      Result := poly.NumVertex - 1
    else
      Result := n;
  end;
  procedure addEdge( vertex: TPoint );
  begin
{ Get the slope of the line from v to i and create an edge record }
    edge := TEdge.Create;
    xDiff := poly.pts[v].X - vertex.X;
    edge.X := poly.pts[v].X;
    edge.YCutoff := vertex.Y;
    edge.Increment := xDiff / ( vertex.Y - y_ );
    lstEdges.Add( edge );
  end;
begin
  rectCanvas := ClipRect;
{ Determine the upper and lower bounds of the polygon }
  vc := poly.NumVertex - 1;
  miny := High( integer );
  maxy := Low( integer );
  for i := 0 to vc do
  begin
    y_ := poly.pts[i].Y;
    if y_ < miny then
      miny := y_;
    if y_ > maxy then
      maxy := y_;
  end;
{ Empty the edge table }
  clearEdgeTable;
{ Begin scanning rows of the polygon, and add edges accordingly }
  for y_ := miny to maxy do
  begin
{ See if any new edges need to be added to the edge table }
    for v := 0 to vc do
      if poly.pts[v].Y = y_ then
{ Add this edge! }
      begin
{ Find the adjacent vertex that is "below" the current vertex }
        i := getAdjPoint( v - 1 );
        if poly.pts[i].Y > y_ then
          addEdge( poly.pts[i] );
        i := getAdjPoint( v + 1 );
        if poly.pts[i].Y > y_ then
          addEdge( poly.pts[i] );
      end;
{ If Y has reached the cutoff point, remove the edge }
     for i := lstEdges.Count - 1 downto 0 do
       if TEdge( lstEdges[i] ).YCutoff = y_ then
       begin
         TEdge( lstEdges[i] ).Free;
         lstEdges.Delete( i );
       end;      
{ Sort the edges by X }
     bSorted := false;
     while not bSorted do
     begin
       bSorted := true;
       for i := 0 to lstEdges.Count - 2 do
       begin
         e1 := TEdge( lstEdges[i] );
         e2 := TEdge( lstEdges[i + 1] );
         if e1.X > e2.X then
         begin
           lstEdges[i] := e2;
           lstEdges[i + 1] := e1;
           bSorted := false;
         end;
       end;
     end;
{ Render lines from consecutive edges }
     i := 0;
     while i < lstEdges.Count do
     begin
       e1 := TEdge( lstEdges[i] );
       linex1 := X + Round(e1.X);
       liney1 := Y + y_;
       e2 := TEdge( lstEdges[i + 1] );
       linex2 := X + Round(e2.X);
       liney2 := liney1;
       if ClipLine( linex1, liney1, linex2, liney2, rectCanvas ) then
       begin
         pOut := Bits;
         Inc( pOut, liney1 * Width + linex1 );
         FillMem( pOut, (linex2 - linex1 + 1), BrushColorIndex );
       end;
       e1.X := e1.X - e1.Increment;
       e2.X := e2.X - e2.Increment;
       Inc( i, 2 );
     end;
  end;
end;

(*********************************************
Render a wireframe, 2D polygon.
*********************************************)
procedure TDIBCanvas.polygon( X, Y: integer; poly: TPolygon );
var
  ptFrom, ptTo: TPoint;
  i: integer;
begin
  ptFrom := poly.pts[0];
  Inc( ptFrom.X, X );
  Inc( ptFrom.Y, Y );
  MoveTo( ptFrom.X, ptFrom.Y );
  for i := 1 to poly.NumVertex - 1 do
  begin
    ptTo := poly.pts[i];
    Inc( ptTo.X, X );
    Inc( ptTo.Y, Y );
    LineTo( ptTo.X, ptTo.Y );
  end;
  LineTo( ptFrom.X, ptFrom.Y );
end;

procedure TDIBCanvas.clearEdgeTable;
begin
  while lstEdges.Count > 0 do
  begin
    TEdge( lstEdges[0] ).Free;
    lstEdges.Delete( 0 );
  end;
end;

procedure TDIBCanvas.Draw( X, Y: integer; DIBCanvas: TDIBCanvas );
var
  rectDest: TRect;
begin
  with rectDest do
  begin
    Left := X;
    Top := Y;
    Right := X + DIBCanvas.Width;
    Bottom := Y + DIBCanvas.Height;
  end;
  CopyRect( rectDest, DIBCanvas, DIBCanvas.ClipRect, true );
end;

end.
